#ifndef network_h
#define network_h

#include <sstream>
#include <boost/lexical_cast.hpp>
#include "input.h"
#include "game.h"

#define MAX_ONLINE_PLAYER 8

namespace exception {

#ifdef EXCEPTION_ENABLE_NETPLAY 
  using boost::asio::ip::tcp;
  typedef shared_ptr<tcp::iostream> socket_ptr;
#endif // EXCEPTION_ENABLE_NETPLAY 

  size_t GetSessionID();


  class NetworkError : public std::runtime_error
  {
  typedef std::runtime_error Super;
  public:
    NetworkError(const string& message) : Super(message) {}
  };


  struct NMessage
  {
    enum {
      UNKNOWN = 0,
      QUERY,
      RESPONSE,
      ENTRY,
      ACCEPT,
      REJECT,
      CLOSE,
      START,
      END,
      PAUSE,
      RESUME,
      STATE,
      JOIN,
      LEAVE,
      CSTAT,
      INPUT,
      TEXT,
      DELAY,
    };

    struct CStatField {
      size_t ping;
    };

    struct StartField {
      int mode;
      int stage;
      int seed;
      int delay;
      int horde_wave;
      int deathmatch_time;
      float teamfortress_life;
    };

    struct InputField {
      int input;
      size_t elapsed;
    };

    struct EntryField {
      char name[16];
      float color[4];
      size_t frame;
    };

    struct TextField {
      char text[64];
    };

    struct JoinField {
      size_t frame;
    };

    struct LeaveField {
      size_t frame;
    };

    struct PauseField {
      size_t frame;
    };

    struct ResumeField {
      size_t frame;
    };

    struct StateField {
      size_t frame;
    };

    struct DelayField {
      size_t delay;
    };


    int type;
    size_t sid;
    union {
      CStatField cstat;
      StartField start;
      InputField input;
      EntryField entry;
      TextField reject;
      TextField response;
      TextField text;
      TextField query;
      JoinField join;
      LeaveField leave;
      PauseField pause;
      ResumeField resume;
      StateField state;
      DelayField delay;
    };
    boost::shared_ptr<ist::bbuffer> statep;

    NMessage() {}
    NMessage(ist::bstream& b) { deserialize(b); }

    void serialize(ist::bstream& b) const
    {
      b << type;
      switch(type) {
      case INPUT:    b << sid << input.input << input.elapsed; break;
      case CSTAT:    b << sid << cstat.ping; break;
      case TEXT:     b << sid << text.text; break;
      case PAUSE:    b << sid << pause.frame; break;
      case RESUME:   b << sid << resume.frame; break;
      case START:    b << sid << start.mode << start.stage << start.seed << start.delay
                       << start.horde_wave << start.deathmatch_time << start.teamfortress_life; break;
      case END:      b << sid; break;
      case CLOSE:    b << sid; break;
      case ACCEPT:   b << sid; break;
      case ENTRY:    b << sid << entry.name << entry.color << entry.frame; break;
      case REJECT:   b << sid << reject.text; break;
      case QUERY:    b << sid << query.text; break;
      case RESPONSE: b << sid << response.text; break;
      case JOIN:     b << sid << join.frame; break;
      case LEAVE:    b << sid << leave.frame; break;
      case DELAY:    b << sid << delay.delay; break;
      case STATE:    b << *statep << state.frame; break;
      default: throw NetworkError("protocol error"); break;
      }
    }

    void deserialize(ist::bstream& b)
    {
      b >> type;
      switch(type) {
      case INPUT:    b >> sid >> input.input >> input.elapsed; break;
      case CSTAT:    b >> sid >> cstat.ping; break;
      case TEXT:     b >> sid >> text.text; break;
      case PAUSE:    b >> sid >> pause.frame; break;
      case RESUME:   b >> sid >> resume.frame; break;
      case START:    b >> sid >> start.mode >> start.stage >> start.seed >> start.delay
                       >> start.horde_wave >> start.deathmatch_time >> start.teamfortress_life; break;
      case END:      b >> sid; break;
      case CLOSE:    b >> sid; break;
      case ACCEPT:   b >> sid; break;
      case ENTRY:    b >> sid >> entry.name >> entry.color >> entry.frame; break;
      case REJECT:   b >> sid >> reject.text; break;
      case QUERY:    b >> sid >> query.text; break;
      case RESPONSE: b >> sid >> response.text; break;
      case JOIN:     b >> sid >> join.frame; break;
      case LEAVE:    b >> sid >> leave.frame; break;
      case DELAY:    b >> sid >> delay.delay; break;
      case STATE:    statep.reset(new Deserializer()); b >> *statep >> state.frame; break;
      default: throw NetworkError("protocol error"); break;
      }
    }


    static NMessage Input(size_t sid, int stat, size_t elapsed)
    {
      NMessage t;
      t.type = INPUT;
      t.sid = sid;
      t.input.input = stat;
      t.input.elapsed = elapsed;
      return t;
    }

    static NMessage Start(GameOption& opt)
    {
      NMessage t;
      t.type = START;
      t.sid = 0;
      t.start.mode = opt.mode;
      t.start.stage = opt.stage;
      t.start.seed = opt.seed;
      t.start.delay = opt.delay;
      t.start.horde_wave = opt.horde_wave;
      t.start.deathmatch_time = opt.deathmatch_time;
      t.start.teamfortress_life = opt.teamfortress_life;
      return t;
    }

    static NMessage End()
    {
      NMessage t;
      t.type = END;
      t.sid = 0;
      return t;
    }

    static NMessage Pause()
    {
      NMessage t;
      t.type = PAUSE;
      t.sid = 0;
      t.pause.frame = GetPast();
      return t;
    }

    static NMessage Resume()
    {
      NMessage t;
      t.type = RESUME;
      t.sid = 0;
      t.resume.frame = GetPast();
      return t;
    }

    static NMessage Close(size_t sid)
    {
      NMessage t;
      t.type = CLOSE;
      t.sid = sid;
      return t;
    }

    static NMessage ClientStatus(size_t sid, size_t ping)
    {
      NMessage t;
      t.type = CSTAT;
      t.sid = sid;
      t.cstat.ping = ping;
      return t;
    }

    static NMessage Text(size_t sid, string text)
    {
      if(text.size()>63) {
        text.resize(63);
      }
      for(size_t i=0; i<text.size(); ++i) {
        if(text[i]=='\'') { text[i]='"'; }
      }

      NMessage t;
      t.type = TEXT;
      t.sid = sid;
      strcpy(t.text.text, text.c_str());
      return t;
    }

    static NMessage Accept(size_t sid)
    {
      NMessage t;
      t.type = ACCEPT;
      t.sid = sid;
      return t;
    }

    static NMessage Reject(const string& text)
    {
      NMessage t;
      t.type = REJECT;
      strcpy(t.reject.text, text.c_str());
      return t;
    }

    static NMessage Entry(size_t sid, const string& name, const vector4& color, size_t frame=0)
    {
      NMessage t;
      t.type = ENTRY;
      t.sid = sid;
      strcpy(t.entry.name, name.c_str());
      t.entry.color[0] = color.x;
      t.entry.color[1] = color.y;
      t.entry.color[2] = color.z;
      t.entry.color[3] = color.w;
      t.entry.frame = frame;
      return t;
    }

    static NMessage Query(const string& text)
    {
      NMessage t;
      t.type = QUERY;
      strcpy(t.query.text, text.c_str());
      return t;
    }

    static NMessage Response(const string& text)
    {
      NMessage t;
      t.type = RESPONSE;
      strcpy(t.response.text, text.c_str());
      return t;
    }

    static NMessage Join(size_t sid, size_t frame=0)
    {
      NMessage t;
      t.type = JOIN;
      t.sid = sid;
      t.join.frame = frame;
      return t;
    }

    static NMessage Leave(size_t sid, size_t frame=0)
    {
      NMessage t;
      t.type = LEAVE;
      t.sid = sid;
      t.leave.frame = frame;
      return t;
    }

    static NMessage State(ist::bbuffer *b, size_t frame)
    {
      NMessage t;
      t.type = STATE;
      t.state.frame = frame;
      t.statep.reset(b);
      return t;
    }

    static NMessage Delay(size_t delay)
    {
      NMessage t;
      t.type = DELAY;
      t.delay.delay = delay;
      return t;
    }
  };

  typedef std::vector<NMessage> nmessage_cont;

#ifdef EXCEPTION_ENABLE_NETPLAY
  inline void SendNetworkMessage(tcp::iostream& socket, const nmessage_cont& mes)
  {
    std::stringstream ss;
    ss << "exception_conflict " << EXCEPTION_VERSION << "\n";
    ist::biostream bio(ss);
    bio << mes.size();
    for(size_t i=0; i<mes.size(); ++i) {
      mes[i].serialize(bio);
    }

    const string& s = ss.str();
    socket.write(s.c_str(), s.size());
  }

  inline void RecvNetworkMessage(tcp::iostream& ss, nmessage_cont& mes)
  {
    string header;
    int version;

    std::getline(ss, header);
    if(sscanf(header.c_str(), "exception_conflict %d", &version)!=1) {
      throw NetworkError("protocol error");
    }
    else if(version!=EXCEPTION_VERSION) {
      throw NetworkError("version not matched");
    }


    size_t size;
    ist::biostream bio(ss);
    bio >> size;
    for(size_t i=0; i<size; ++i) {
      mes.push_back(NMessage(bio));
    }
  }
#endif // EXCEPTION_ENABLE_NETPLAY 

  class Thread
  {
  private:
    typedef boost::shared_ptr<boost::thread> thread_ptr;
    thread_ptr m_thread;
    bool m_is_running;

  public:
    Thread() : m_is_running(false)
    {}

    virtual ~Thread()
    {}

    virtual void run()
    {
      m_thread.reset(new boost::thread(boost::ref(*this)));
    }

    void join()
    {
      if(m_thread) {
        m_thread->join();
      }
    }

    bool isRunning() const
    {
      return m_is_running;
    }

    void operator()()
    {
      m_is_running = true;
      exec();
      m_is_running = false;
    }

    virtual void exec()=0;
  };



  class IInputServer : public Thread
  {
  public:
    virtual void flush()=0;
  };

#ifdef EXCEPTION_ENABLE_NETPLAY
  class InputServer : public IInputServer
  {
  typedef IInputServer Super;
  public:

    class Client : public Thread
    {
    private:
      boost::mutex m_recv_mutex;
      boost::mutex m_send_mutex;
      nmessage_cont m_trecv_data;
      nmessage_cont m_recv_data;
      nmessage_cont m_send_data;
      nmessage_cont m_cstat;
      socket_ptr m_socket;
      const size_t m_session_id;
      const string m_name;
      const vector4 m_color;
      bool m_stopped;
      bool m_closed;
      size_t m_ping;
      int m_input_length;

    public:
      Client(socket_ptr s, size_t sid, const string& name, const vector4& color) :
        m_socket(s), m_session_id(sid), m_name(name), m_color(color), m_stopped(false), m_closed(false), m_ping(0)
      {}

      ~Client()
      {
        stop();
        join();
      }

      bool isClosed() { return m_closed; }
      size_t getID() { return m_session_id; }
      const string& getName() { return m_name; }
      const vector4& getColor() { return m_color; }
      size_t getPing() { return m_ping; }

      void push(const NMessage& m) // sync 
      {
        boost::mutex::scoped_lock lock(m_send_mutex);
        m_send_data.push_back(m);
      }

      void push(const nmessage_cont& mc) // sync 
      {
        boost::mutex::scoped_lock lock(m_send_mutex);
        m_cstat.clear();
        for(size_t i=0; i<mc.size(); ++i) {
          if(mc[i].type==NMessage::CSTAT) {
            m_cstat.push_back(mc[i]);
          }
          else {
            m_send_data.push_back(mc[i]);
          }
        }
      }

      void pop(nmessage_cont& mc) // sync 
      {
        boost::mutex::scoped_lock lock(m_recv_mutex);
        mc.insert(mc.end(), m_recv_data.begin(), m_recv_data.end());
        m_recv_data.clear();
      }

      NMessage waitMessage(int type)
      {
        for(;;) {
          {
            boost::mutex::scoped_lock lock(m_recv_mutex);
            for(nmessage_cont::iterator p=m_recv_data.begin(); p!=m_recv_data.end(); ++p) {
              if(p->type==type) {
                NMessage tmp = *p;
                m_recv_data.erase(p);
                return tmp;
              }
            }
          }
          sgui::Sleep(5);
        }
      }

      void stop()
      {
        m_stopped = true;
      }

      void exec()
      {
        try {
          while(!m_stopped) {
            int t = sgui::GetTicks();

            recv();
            send();

            m_ping = sgui::GetTicks()-t;
            // [JƐŋ肷邽߁Aꍇ҂ 
            if(m_ping < 1) {
              sgui::Sleep(1);
            }
          }
        }
        catch(...) {
        }
        m_closed = true;
        m_socket->close();
      }


    private:
      void send() // sync 
      {
        boost::mutex::scoped_lock lock(m_send_mutex);
        m_send_data.insert(m_send_data.end(), m_cstat.begin(), m_cstat.end());
        SendNetworkMessage(*m_socket, m_send_data);
        m_send_data.clear();
      }

      bool handleMessage(const NMessage& m)
      {
        if(m.type==NMessage::CLOSE || m.type==NMessage::END) {
          stop();
          return true;
        }
        return false;
      }

      struct handler
      {
        Client *m_client;
        handler(Client *c) : m_client(c) {}
        bool operator()(const NMessage& m)
        {
          return m_client->handleMessage(m);
        }
      };
      friend struct handler;

      void recv() // sync 
      {
        RecvNetworkMessage(*m_socket, m_trecv_data);
        for(size_t i=0; i<m_trecv_data.size(); ++i) {
          m_trecv_data[i].sid = m_session_id;
        }
        m_trecv_data.erase(
          std::remove_if(m_trecv_data.begin(), m_trecv_data.end(), handler(this)),
          m_trecv_data.end());

        {
          boost::mutex::scoped_lock lock(m_recv_mutex);
          m_recv_data.insert(m_recv_data.end(), m_trecv_data.begin(), m_trecv_data.end());
        }
        m_trecv_data.clear();
      }
    };


    class Appender : public Thread
    {
    private:
      string m_comment;

    public:
      Appender(const string& comment="") : m_comment(comment)
      {}

      ~Appender()
      {
        join();
      }

      void exec()
      {
        boost::asio::io_service ios;
        ist::HTTPRequest req(ios);

        char url[256];

        IConfig& c = *GetConfig();
        {
          boost::mutex::scoped_lock lock(c.getMutex());
          sprintf(url,
            EXCEPTION_HOST_PATH "server/add&port=%d&name=%s&comment=%s",
            c.port, c.scorename.c_str(), m_comment.c_str());
        }
        req.get(EXCEPTION_HOST, url);
      }
    };

    class Remover : public Thread
    {
    public:
      ~Remover()
      {
        join();
      }

      void exec()
      {
        boost::asio::io_service ios;
        ist::HTTPRequest req(ios);
        req.get(EXCEPTION_HOST, EXCEPTION_HOST_PATH "server/del");
      }
    };

    class Flusher : public Thread
    {
    private:
      IInputServer *m_server;
      volatile bool m_stoped;

    public:
      Flusher(IInputServer& server) :
        m_server(&server), m_stoped(false)
      {}

      ~Flusher()
      {
        stop();
        join();
      }

      void stop()
      {
        m_stoped = true;
      }

      void exec()
      {
        while(!m_stoped) {
          int t = sgui::GetTicks();

          m_server->flush();

          // ɘa 
          int past = sgui::GetTicks()-t;
          if(past<1) {
            sgui::Sleep(1);
          }
        }
      }
    };
    typedef shared_ptr<Thread> thread_ptr;
    typedef shared_ptr<Client> client_ptr;
    typedef std::map<int, client_ptr> client_cont;
    typedef shared_ptr<Flusher> flusher_ptr;


  private:
    boost::asio::io_service &m_io_service;
    boost::mutex m_mutex;
    bool m_stopped;
    client_cont m_clients;
    flusher_ptr m_flusher;
    thread_ptr m_remover;
    nmessage_cont m_send_data;
    int m_delay;
    int m_idgen;

  public:
    InputServer(boost::asio::io_service& ios) :
      m_io_service(ios), m_stopped(false), m_delay(6), m_idgen(0)
    {
    }

    ~InputServer()
    {
      stop();
      join();

      closeClients();
      m_flusher.reset();
    }

    void stop()
    {
      if(m_stopped) {
        return;
      }
      m_stopped = true;

      tcp::iostream socket("127.0.0.1", boost::lexical_cast<string>(GetConfig()->port));
      nmessage_cont sm;
      SendNetworkMessage(socket, sm);
    }

    void exec()
    {
      m_flusher.reset(new Flusher(*this));
      m_flusher->run();

      tcp::acceptor acceptor(m_io_service, tcp::endpoint(tcp::v4(), GetConfig()->port));
      while(!m_stopped) {
        socket_ptr s(new tcp::iostream());
        acceptor.accept(*s->rdbuf());
        s->rdbuf()->set_option(tcp::no_delay(true));
        if(!m_stopped) {
          accept(s);
        }
      }

      m_remover.reset(new Remover());
      m_remover->run();
    }

    void accept(socket_ptr& socket)
    {
      nmessage_cont rm;
      try {
        RecvNetworkMessage(*socket, rm);
      }
      catch(const NetworkError& e) {
        nmessage_cont sm;
        sm.push_back(NMessage::Reject(e.what()));
        SendNetworkMessage(*socket, sm);
        return;
      }

      for(size_t i=0; i<rm.size(); ++i) {
        NMessage& m = rm[i];
        if(m.type==NMessage::ENTRY) { // Q]NGXgꍇ 
          processEntry(socket, m);
        }
        else if(m.type==NMessage::QUERY) {
          processQuery(socket, m);
        }
      }
    }

    void processEntry(socket_ptr& socket, NMessage& m)
    {
      boost::mutex::scoped_lock lock(m_mutex); // sync 
      nmessage_cont sm;

      // őlĂreject 
      if(m_clients.size()>=MAX_ONLINE_PLAYER) {
        char buf[128];
        sprintf(buf, "too many players (max: %dplayers)", MAX_ONLINE_PLAYER);
        sm.push_back(NMessage::Reject(buf));
        SendNetworkMessage(*socket, sm);
        return;
      }
      // 100msȓɉԂĂȂreject 
      if(getResponseTime(socket)>=100) {
        sm.push_back(NMessage::Reject("network too lazy"));
        SendNetworkMessage(*socket, sm);
        return;
      }

      // ID蓖 
      int sid = ++m_idgen;
      client_ptr c(new Client(socket, sid, m.entry.name, vector4(m.entry.color)));
      m_clients[sid] = c;
      sm.push_back(NMessage::Accept(sid));

      // rQ̏ꍇAXe[gƓrQbZ[W𑗐M 
      if(IGame *game = GetGame()) {
        // incoming new player\郁bZ[WSNCAgɑM 
        for(client_cont::iterator p=m_clients.begin(); p!=m_clients.end(); ++p) {
          client_ptr& cli = p->second;
          NMessage m = NMessage::Response("new player");
          m.sid = sid;
          cli->push(m);
        }

        client_ptr local = m_clients[GetSessionID()];
        local->push(NMessage::Query("state"));
        NMessage m = local->waitMessage(NMessage::STATE);
        sm.push_back(m);

        m_send_data.push_back(NMessage::Entry(c->getID(), c->getName(), c->getColor(), m.state.frame+m_delay));
        m_send_data.push_back(NMessage::Join(c->getID(), m.state.frame+m_delay));
      }
      else {
        // ̃vC[̏𑗐M 
        for(client_cont::iterator p=m_clients.begin(); p!=m_clients.end(); ++p) {
          client_ptr& cli = p->second;
          sm.push_back(NMessage::Entry(cli->getID(), cli->getName(), cli->getColor()));
        }
        m_send_data.push_back(NMessage::Entry(c->getID(), c->getName(), c->getColor()));
      }
      SendNetworkMessage(*socket, sm);
      c->run();
    }

    size_t getResponseTime(socket_ptr& socket)
    {
      nmessage_cont mc;
      Uint32 start = sgui::GetTicks();
      SendNetworkMessage(*socket, mc);
      RecvNetworkMessage(*socket, mc);
      Uint32 t = sgui::GetTicks()-start;
      printf("response time: %d\n", t);
      return t;
    }


    void processQuery(socket_ptr& socket, NMessage& m)
    {
      nmessage_cont sm;
      string query(m.query.text);
      if(query=="version") {
        sm.push_back(NMessage::Response(string("exception_conflict ")+boost::lexical_cast<string>(EXCEPTION_VERSION)));
        SendNetworkMessage(*socket, sm);
      }
    }



    void flush() // sync 
    {
      boost::mutex::scoped_lock lock(m_mutex);

      // NCAgɗ܂bZ[W܂Ƃ߂ 
      for(client_cont::iterator p=m_clients.begin(); p!=m_clients.end(); ++p) {
        p->second->pop(m_send_data);
      }
      for(size_t a=0; a<m_send_data.size(); ++a) {
        NMessage& m = m_send_data[a];
        if(m.type==NMessage::PAUSE) {
          m.pause.frame+=m_delay;
        }
        else if(m.type==NMessage::DELAY) {
          m_delay = m.delay.delay+2;
        }
      }

      // NCAg̏ԂbZ[W 
      for(client_cont::iterator p=m_clients.begin(); p!=m_clients.end(); /**/) {
        client_ptr& c = p->second;
        int sid = p->first;
        if(c->isClosed()) { // RlNV؂Ă 
          if(IGame *game = GetGame()) {
            if(sid==GetSessionID()) {
              m_send_data.push_back(NMessage::End());
            }
            else {
              m_send_data.push_back(NMessage::Leave(sid, game->getPast()+m_delay));
            }
          }
          m_send_data.push_back(NMessage::Close(sid));
          m_clients.erase(p++);
        }
        else {
          m_send_data.push_back(NMessage::ClientStatus(sid, c->getPing()));
          ++p;
        }
      }

      // SbZ[WM 
      for(client_cont::iterator p=m_clients.begin(); p!=m_clients.end(); ++p) {
        p->second->push(m_send_data);
      }
      m_send_data.clear();
    }

    void closeClients() // sync 
    {
      boost::mutex::scoped_lock lock(m_mutex);
      for(client_cont::iterator p=m_clients.begin(); p!=m_clients.end(); ++p) {
        p->second->stop();
      }
      m_clients.clear();
    }
  };
#endif // EXCEPTION_ENABLE_NETPLAY 



  class Session : public ISession
  {
  private:
    input_ptr m_input;
    size_t m_id;
    string m_name;
    vector4 m_color;
    size_t m_ping;
    size_t m_delay;
    size_t m_elapsed;
    size_t m_begin_frame;
    size_t m_join_frame;
    size_t m_leave_frame;

  public:
    Session(Deserializer& s)
    {
      m_input = new InputStream(s);
      s >> m_id >> m_name >> m_color >> m_color >> m_ping >> m_delay>> m_elapsed
        >> m_begin_frame >> m_join_frame >> m_leave_frame;
    }

    void serialize(Serializer& s) const
    {
      m_input->serialize(s);
      s << m_id << m_name << m_color << m_color << m_ping << m_delay<< m_elapsed
        << m_begin_frame << m_join_frame << m_leave_frame;
    }

  public:
    Session(size_t id, const string& name, const vector4& color, input_ptr input) :
      m_input(input), m_id(id), m_name(name), m_color(color), m_ping(0), m_delay(5), m_elapsed(0),
      m_begin_frame(0), m_join_frame(0), m_leave_frame(-1)
    {}

    Session(ist::gzbstream& s, int version) :
      m_id(0), m_ping(0), m_elapsed(0), m_begin_frame(0), m_join_frame(0), m_leave_frame(-1)
    {
      s >> m_name;
      m_input = new InputStream(s);
      s >> m_color >> m_id >> m_begin_frame >> m_join_frame >> m_leave_frame;
    }

    void write(ist::gzbstream& s) const
    {
      s << m_name;
      m_input->write(s);
      s << m_color << m_id << m_begin_frame << m_join_frame << m_leave_frame;
    }

    size_t getID() const { return m_id; }
    const string& getName() const { return m_name; }
    const vector4& getColor() const { return m_color; }
    InputStream* getInput() { return m_input.get(); }

    size_t getPing() const       { return m_ping; }
    size_t getDelay() const      { return m_delay; }
    size_t getElapsed() const    { return m_elapsed; }
    size_t getBeginFrame() const { return m_begin_frame; }
    size_t getJoinFrame() const  { return m_join_frame; }
    size_t getLeaveFrame() const { return m_leave_frame; }
    void setPing(size_t v)       { m_ping=v; }
    void setDelay(size_t v)      { m_delay=v; }
    void setElapsed(size_t v)    { m_elapsed=v; }
    void setBeginFrame(size_t v) { m_begin_frame=v; }
    void setJoinFrame(size_t v)  { m_join_frame=v; }
    void setLeaveFrame(size_t v) { m_leave_frame=v; }

    void setID(size_t v) { m_id=v; }
  };
  typedef intrusive_ptr<Session> session_ptr;




  class IInputClient : public Thread
  {
  public:
    virtual void serialize(Serializer& s)=0;
    virtual void deserialize(Deserializer& s)=0;
    virtual void write(ist::gzbstream& s)=0;
    virtual size_t getSessionCount()=0;
    virtual session_ptr getSession(size_t i)=0;
    virtual session_ptr getSessionByID(size_t id)=0;
    virtual size_t getSessionID()=0;
    virtual int getDelay()=0;
    virtual void push(const NMessage& m)=0;
    virtual void flush()=0;
    virtual void sync()=0;
    virtual void update()=0;
  };

  class BaseInputClient : public IInputClient
  {
  protected:
    typedef std::map<size_t, session_ptr> session_cont;
    session_cont m_session;

  public:
    virtual void serialize(Serializer& s)
    {
      s << m_session.size();
      for(session_cont::const_iterator p=m_session.begin(); p!=m_session.end(); ++p) {
        p->second->serialize(s);
      }
    }

    virtual void deserialize(Deserializer& s)
    {
      size_t size;
      s >> size;
      for(size_t i=0; i<size; ++i) {
        session_ptr session(new Session(s));
        m_session[session->getID()] = session;
      }
    }

    virtual void write(ist::gzbstream& s)
    {
      s << m_session.size();
      for(session_cont::iterator p=m_session.begin(); p!=m_session.end(); ++p) {
        p->second->write(s);
      }
    }

    virtual size_t getSessionCount()
    {
      return m_session.size();
    }

    virtual session_ptr getSession(size_t i)
    {
      if(i>=m_session.size()) {
        return session_ptr();
      }
      session_cont::iterator p = m_session.begin();
      std::advance(p, i);
      return p->second;
    }

    virtual session_ptr getSessionByID(size_t id)
    {
      session_cont::iterator p = m_session.find(id);
      if(p==m_session.end()) {
        return session_ptr();
      }
      return p->second;
    }

    virtual session_ptr findSession(size_t id)
    {
      session_cont::iterator p = m_session.find(id);
      return (p==m_session.end()) ? session_ptr() : p->second;
    }

    virtual size_t getSessionID()=0;
    virtual int getDelay() { return 0; }
    virtual void push(const NMessage& m)=0;
    virtual void flush() {}
    virtual void sync() {}

    virtual void update()
    {
      for(session_cont::iterator p=m_session.begin(); p!=m_session.end(); ++p) {
        p->second->getInput()->update();
      }
    }

    virtual void exec() {}
  };



  class InputClientLocal : public BaseInputClient
  {
  public:
    InputClientLocal()
    {
      session_ptr s(
        new Session(0, GetConfig()->scorename, GetConfig()->color, new InputStream()));
      m_session[s->getID()] = s;
    }

    size_t getSessionID()
    {
      return 0;
    }

    void push(const NMessage& m)
    {
      int t = m.type;
      if(t==NMessage::INPUT) {
        getSessionByID(m.sid)->getInput()->push(m.input.input);
      }
      else if(t==NMessage::PAUSE) {
        Pause();
      }
      else if(t==NMessage::RESUME){
        Resume();
      }
      else if(t==NMessage::END)   {
        FadeToTitle();
      }
    }
  };


  class InputClientReplay : public BaseInputClient
  {
  typedef BaseInputClient Super;
  private:
    session_cont m_join;
    size_t m_length;

  public:
    InputClientReplay(ist::gzbstream& s, int version) : m_length(0)
    {
      size_t num;
      s >> num;
      for(int i=0; i<num; ++i) {
        session_ptr session(new Session(s, version));
        if(i==0) {
          m_length = session->getBeginFrame()+session->getInput()->getLength();
        }
        if(session->getBeginFrame()==0) {
          m_session[session->getID()] = session;
        }
        else {
          m_join[session->getID()] = session;
        }
      }
    }

    InputClientReplay() : m_length(0)
    {}

    virtual void serialize(Serializer& s)
    {
      Super::serialize(s);
      s << m_join.size();
      for(session_cont::const_iterator p=m_join.begin(); p!=m_join.end(); ++p) {
        p->second->serialize(s);
      }
      s << m_length;
    }

    virtual void deserialize(Deserializer& s)
    {
      Super::deserialize(s);
      size_t size;
      s >> size;
      for(size_t i=0; i<size; ++i) {
        session_ptr session(new Session(s));
        m_join[session->getID()] = session;
      }
      s >> m_length;
    }

    size_t getLength() { return m_length; }
    size_t getSessionID() { return 0; }

    void push(const NMessage& m)
    {
      int t = m.type;
      if(t==NMessage::PAUSE) {
        Pause();
      }
      else if(t==NMessage::RESUME){
        Resume();
      }
      else if(t==NMessage::END)   {
        FadeToTitle();
      }
    }

    void sync()
    {
      IGame *game = GetGame();
      if(!game) {
        return;
      }
      for(session_cont::iterator p=m_join.begin(); p!=m_join.end();/**/) {
        session_ptr s = p->second;
        if(s->getBeginFrame()==game->getPast()) {
          m_session[s->getID()] = s;
          m_join.erase(p++);
        }
        else {
          ++p;
        }
      }
      for(session_cont::iterator p=m_session.begin(); p!=m_session.end();/**/) {
        session_ptr s = p->second;
        if(s->getJoinFrame()==game->getPast()) {
          game->join(s->getID());
        }
        if(s->getLeaveFrame()==game->getPast()) {
          game->leave(s->getID());
          m_session.erase(p++);
        }
        else {
          ++p;
        }
      }
    }
  };


#ifdef EXCEPTION_ENABLE_NETPLAY 
  IInputServer* GetInputServer();

  class InputClientIP : public BaseInputClient
  {
  typedef BaseInputClient Super;
  private:
    socket_ptr m_socket;
    boost::recursive_mutex m_recv_mutex;
    boost::mutex m_send_mutex;
    nmessage_cont m_trecv_data;
    nmessage_cont m_recv_data;
    nmessage_cont m_send_data;
    nmessage_cont m_start_data;
    session_cont m_closed;
    bool m_stopped;
    size_t m_session_id;
    size_t m_delay;
    bool m_request_state;
    int m_wait_count;
    int m_nowait_count;

    GameOption m_opt;
    bool m_started;

  public:
    InputClientIP() :
        m_stopped(false), m_session_id(0), m_delay(5),
        m_request_state(false), m_wait_count(0), m_nowait_count(0),
        m_started(false)
    {}

    ~InputClientIP()
    {
      stop();
      join();
    }

    void serialize(Serializer& s)
    {
      Super::serialize(s);
      s << m_closed.size();
      for(session_cont::const_iterator p=m_closed.begin(); p!=m_closed.end(); ++p) {
        p->second->serialize(s);
      }
      s << m_delay;

      {
        boost::recursive_mutex::scoped_lock lock(m_recv_mutex);
        s << m_recv_data.size();
        for(size_t i=0; i<m_recv_data.size(); ++i) {
          m_recv_data[i].serialize(s);
        }
      }
    }

    void deserialize(Deserializer& s)
    {
      Super::deserialize(s);
      size_t size;
      s >> size;
      for(size_t i=0; i<size; ++i) {
        session_ptr s(new Session(s));
        m_closed[s->getID()] = s;
      }
      s >> m_delay;

      boost::recursive_mutex::scoped_lock lock(m_recv_mutex);
      s >> size;
      for(size_t i=0; i<size; ++i) {
        m_recv_data.push_back(NMessage(s));
      }
    }

    void write(ist::gzbstream& s)
    {
      session_cont tmp;
      tmp.insert(m_session.begin(), m_session.end());
      tmp.insert(m_closed.begin(), m_closed.end());

      s << tmp.size();
      for(session_cont::iterator p=tmp.begin(); p!=tmp.end(); ++p) {
        p->second->write(s);
      }
    }


    int getDelay() { return m_delay; }
    size_t getSessionID() { return m_session_id; }

    virtual session_ptr getSessionByID(size_t id)
    {
      if(session_ptr s = Super::getSessionByID(id)) {
        return s;
      }
      session_cont::iterator p = m_closed.find(id);
      if(p==m_closed.end()) {
        return session_ptr();
      }
      return p->second;
    }

    void stop() // sync 
    {
      m_stopped = true;
    }

    void run()
    {
      for(size_t i=0; i<m_start_data.size(); ++i) {
        dispatch(m_start_data[i]);
      }
      m_start_data.clear();
      Super::run();
    }

    void connect(const string& host, ushort port)
    {
      m_socket.reset(new tcp::iostream(host, boost::lexical_cast<string>(port)));
      if(!*m_socket) {
        throw NetworkError("connection failed");
      }
      m_socket->rdbuf()->set_option(tcp::no_delay(true));

      {
        IConfig& conf = *GetConfig();
        boost::mutex::scoped_lock lock(conf.getMutex());

        nmessage_cont sm;
        sm.push_back(NMessage::Entry(0, conf.scorename, conf.color));
        SendNetworkMessage(*m_socket, sm);
      }
      {
        nmessage_cont mc;
        RecvNetworkMessage(*m_socket, m_start_data);
        SendNetworkMessage(*m_socket, mc);
      }
      {
        RecvNetworkMessage(*m_socket, m_start_data);
        for(size_t i=0; i<m_start_data.size(); ++i) {
          NMessage& m = m_start_data[i];
          if(m.type==NMessage::ACCEPT) {
            m_session_id = m.sid;
          }
          else if(m.type==NMessage::REJECT) {
            throw NetworkError(string("rejected: ")+m.reject.text);
          }
        }
      }
    }


    void push(const NMessage& t) // sync 
    {
      if(t.type==NMessage::INPUT && !findSession(getSessionID())) {
        return;
      }
      boost::mutex::scoped_lock lock(m_send_mutex);
      m_send_data.push_back(t);
    }


    struct handler
    {
      InputClientIP *client;
      handler(InputClientIP *c) : client(c) {}
      bool operator()(NMessage& m) {
        return client->dispatch(m);
      }
    };
    friend struct handler;

    struct closer
    {
      std::vector<int> closed;
      void selectClosed(nmessage_cont& nc)
      {
        for(size_t i=0; i<nc.size(); ++i) {
          if(nc[i].type==NMessage::CLOSE) {
            closed.push_back(nc[i].sid);
          }
        }
      }
      bool operator()(NMessage& m) {
        return m.type==NMessage::ENTRY && std::find(closed.begin(), closed.end(), m.sid)!=closed.end();
      }
    };

    void flush() // sync 
    {
      boost::recursive_mutex::scoped_lock lock(m_recv_mutex);

      closer closed;
      closed.selectClosed(m_recv_data);
      m_recv_data.erase(std::remove_if(m_recv_data.begin(), m_recv_data.end(), handler(this)), m_recv_data.end());
      m_recv_data.erase(std::remove_if(m_recv_data.begin(), m_recv_data.end(), closed), m_recv_data.end());

      if(m_request_state) {
        m_request_state = false;
        if(IGame *game = GetGame()) {
          Serializer *seri = new Serializer();
          game->serialize(*seri);
          push(NMessage::State(seri, game->getPast()));
        }
      }
    }


    bool needSync()
    {
      if(!GetGame()) {
        return false;
      }
      for(session_cont::iterator p=m_session.begin(); p!=m_session.end(); ++p) {
        InputStream& input = *p->second->getInput();
        if(input.getIndex()>=input.getLength()) {
          return true;
        }
      }
      return false;
    }

    void sync()
    {
      flush();

      // T[o[vC[fBCL̏ꍇA҂̕pxdelay𒲐߂ 
      IGame *game = GetGame();
      if(IsServerMode() && GetConfig()->autodelay && (game && !game->isPaused())) {
        bool delayed = false;
        bool overwork = false;
        size_t ping = 0;
        for(session_cont::iterator p=m_session.begin(); p!=m_session.end(); ++p) {
          session_ptr s = p->second;
          ping = std::max<size_t>(ping, s->getPing());
          if(s->getElapsed()<16 || s->getElapsed()>17) {
            overwork = true;
          }
          InputStream& input = *s->getInput();
          if(input.getIndex()>=input.getLength()) { // ҂Kv 
            delayed = true;
          }
        }

        if(delayed) {
          if(ping>=m_delay*5) {
            m_wait_count+=15;
            m_nowait_count = std::max<int>(0, m_nowait_count-5);
          }
          else if(overwork) {
            m_wait_count = std::max<int>(0, m_wait_count-1);
          }
          else {
            m_wait_count+=5;
          }
          if(m_wait_count>=100) {
            m_wait_count = 0;
            m_nowait_count = 0;
            push(NMessage::Delay(m_delay+1));
          }
        }
        else {
          m_nowait_count+=1;
          m_wait_count = std::max<int>(0, m_wait_count-1);
          if(m_nowait_count>200 && m_delay>1) {
            m_nowait_count = 0;
            m_wait_count = 90;
            push(NMessage::Delay(m_delay-1));
          }
        }
      }

      // xĂpeer҂ 
      while(needSync() && isRunning()) {
        sgui::Sleep(1);
        flush();
      }
    }

  private:
    bool dispatch(const NMessage& m)
    {
      int t = m.type;
      if(t==NMessage::INPUT) {
        if(session_ptr s = findSession(m.sid)) {
          s->getInput()->push(m.input.input);
          s->setElapsed(m.input.elapsed);
        }
        else {
        //  throw Error("NMessage::INPUT");
        }
      }
      else if(t==NMessage::CSTAT) {
        if(session_ptr s = findSession(m.sid)) {
          s->setPing(m.cstat.ping);
        }
      }
      else if(t==NMessage::STATE) {
        LoadState(dynamic_cast<Deserializer&>(*m.statep));
        push(NMessage::Response("state restored"));
      }
      else if(t==NMessage::ENTRY) {
        if(IGame *game = GetGame()) {
          if(m.entry.frame > game->getPast()) {
            return false;
          }
          else if(m.entry.frame < game->getPast()) {
            throw Error("NMessage::ENTRY");
          }
        }
        else if(m_started && IsServerMode()) {
          m_opt.delay = m_delay;
          push(NMessage::Start(m_opt));
        }

        if(!findSession(m.sid)) {
          session_ptr s(
            new Session(m.sid, m.entry.name, vector4(m.entry.color), new InputStream()));
          m_session[m.sid] = s;
          PushChatText(string("# ")+s->getName()+string(" join"));
          if(IGame *game = GetGame()) {
            s->setBeginFrame(m.entry.frame);
            s->setJoinFrame(-1);
            for(int j=0; j<m_delay; ++j) {
              s->getInput()->push(0);
            }
          }
          else {
            GetSound("charge1.wav")->play(6);
            s->getInput()->resize(m_opt.delay);
            if(IsServerMode()) {
              char buf[128];
              sprintf(buf, "%dplayers", m_session.size());
              UpdateServerInfomation(buf);
            }
          }
        }
      }
      else if(t==NMessage::JOIN) {
        if(IGame *game = GetGame()) {
          if(m.join.frame > game->getPast()) {
            return false;
          }
          else if(m.join.frame < game->getPast()) {
            throw Error("NMessage::JOIN");
          }
          else if(m.join.frame==game->getPast()) {
            if(session_ptr s=findSession(m.sid)) {
              s->setJoinFrame(m.join.frame);
              game->join(m.sid);
            }
          }
        }
      }
      else if(t==NMessage::LEAVE) {
        if(IGame *game = GetGame()) {
          if(m.leave.frame > game->getPast()) {
            return false;
          }
          else if(m.leave.frame < game->getPast()) {
            throw Error("NMessage::LEAVE");
          }
          else if(m.leave.frame==game->getPast()) {
            session_ptr s;
            session_cont::iterator i = m_closed.find(m.sid);
            if(i!=m_closed.end()) {
              s = i->second;
            }
            else {
              s = findSession(m.sid);
            }
            if(s) {
              game->leave(m.sid);
              s->setLeaveFrame(m.leave.frame);
            }
          }
        }
      }
      else if(t==NMessage::TEXT)  {
        if(session_ptr s = findSession(m.sid)) {
          PushChatText(string(s->getName())+": "+m.text.text);
        }
      }
      else if(t==NMessage::PAUSE) {
        if(IGame *game = GetGame()) {
          if(m.pause.frame > game->getPast()) {
            return false;
          }
          else if(m.pause.frame<=game->getPast()) {
            Pause();
          }
        }
      }
      else if(t==NMessage::RESUME){
        if(IGame *game = GetGame()) {
          if(m.resume.frame > game->getPast()) {
            return false;
          }
          else if(m.pause.frame<=game->getPast()) {
            Resume();
          }
        }
      }
      else if(t==NMessage::START) {
        if(m_started) {
          return true;
        }
        m_started = true;
        m_opt.mode = m.start.mode;
        m_opt.stage = m.start.stage;
        m_opt.seed = m.start.seed;
        m_opt.horde_wave = m.start.horde_wave;
        m_opt.deathmatch_time = m.start.deathmatch_time;
        m_opt.teamfortress_life = m.start.teamfortress_life;
        m_opt.delay = m_delay = m.start.delay;
        for(session_cont::iterator i=m_session.begin(); i!=m_session.end(); ++i) {
          i->second->getInput()->resize(m_delay);
        }
        FadeToGame(m_opt);
      }
      else if(t==NMessage::END) {
        if(IGame *game = GetGame()) {
          game->setPause(true);
          FadeToTitle();
        }
        else {
          m_closed.clear();
          m_session.clear();
        }
      }
      else if(t==NMessage::CLOSE) {
        ReleaseWaitingForNewPlayer(m.sid);
        if(session_ptr s = findSession(m.sid)) {
          PushChatText(string("# ")+s->getName()+" disconnected");
          m_session.erase(s->getID());
          if(GetGame()) {
            m_closed[s->getID()] = s;
          }
          else {
            if(IsServerMode()) {
              char buf[128];
              sprintf(buf, "%dplayers", m_session.size());
              UpdateServerInfomation(buf);
            }
          }
        }
      }
      else if(t==NMessage::DELAY) {
        int gap = m.delay.delay-m_delay;
        m_delay = m.delay.delay;
        if(GetGame()) {
          for(session_cont::iterator i=m_session.begin(); i!=m_session.end(); ++i) {
            i->second->getInput()->modify(gap);
          }
        }
      }
      else if(t==NMessage::QUERY)   {
        string query(m.query.text);
        if(query=="state") {
          m_request_state = true;
        }
        else {
          return false;
        }
      }
      else if(t==NMessage::RESPONSE) {
        string mes = m.response.text;
        if(mes=="new player") {
          WaitForNewPlayer(m.sid);
        }
        else if(mes=="state restored") {
          ReleaseWaitingForNewPlayer(m.sid);
        }
      }
      return true;
    }


    void exec()
    {
      try {
        while(!m_stopped) {
          send();
          recv();
        }
      }
      catch(...) {
        boost::recursive_mutex::scoped_lock lock(m_recv_mutex);
        m_recv_data.push_back(NMessage::Text(0, "# connection closed"));
        m_recv_data.push_back(NMessage::End());
      }
      m_socket->close();
    }

    void send() // sync 
    {
      boost::mutex::scoped_lock lock(m_send_mutex);
      SendNetworkMessage(*m_socket, m_send_data);
    //  printf("send() %d\n", m_send_data.size());
      m_send_data.clear();
    }

    void recv() // sync 
    {
      // x؃eXgp 
      /*
      int s_interval = 10;
      if(::time(0)%30<15) {
        s_interval = 40;
      }
      else {
        s_interval = 10;
      }
      sgui::Sleep(s_interval);
      */

      RecvNetworkMessage(*m_socket, m_trecv_data);
    //  printf("recv() %d\n", m_trecv_data.size());
      for(size_t i=0; i<m_trecv_data.size(); ++i) {
        NMessage& m = m_trecv_data[i];
        if(m.type==NMessage::END) {
          m_stopped = true;
        }
      }
      {
        boost::recursive_mutex::scoped_lock lock(m_recv_mutex);
        m_recv_data.insert(m_recv_data.end(), m_trecv_data.begin(), m_trecv_data.end());
      }
      m_trecv_data.clear();
    }
  };
#endif // EXCEPTION_ENABLE_NETPLAY 

  typedef shared_ptr<IInputServer> iserver_ptr;
  typedef shared_ptr<IInputClient> iclient_ptr;

  extern iserver_ptr g_iserver;
  extern iclient_ptr g_iclient;

  IInputClient* GetInputClient();
  void SendNMessage(const NMessage& t);
  size_t GetSessionCount();
  session_ptr GetSession(size_t i);
  session_ptr GetSessionByID(size_t i);

}
#endif
